<?php
/**
 * =======================================================
 *        ULTRA SECURITY SYSTEM – FINAL FIXED VERSION
 *        حماية كاملة لجامعة النبراس إصدار 2025
 * =======================================================
 */


/* =======================================================
    1) تشغيل الجلسة بشكل آمن بدون مشاكل
======================================================= */
function secure_session_start()
{
    // فقط إذا الجلسة غير شغالة
    if (session_status() === PHP_SESSION_NONE) {

        // إعدادات الجلسة يجب أن تُستخدم قبل session_start()
        ini_set('session.use_strict_mode', 1);

        session_set_cookie_params([
            'lifetime' => 0,
            'httponly' => true,
            'secure'   => isset($_SERVER["HTTPS"]),
            'samesite' => 'Strict'
        ]);

        // تشغيل الجلسة
        session_start();

        // أول مرة فقط
        if (!isset($_SESSION['initiated'])) {
            session_regenerate_id(true);
            $_SESSION['initiated'] = true;
        }
    }
}

// تشغيل الجلسة
secure_session_start();



/* =======================================================
    2) حماية XSS – Sanitizing
======================================================= */

function sanitize($data){
    if (is_array($data)) return sanitize_array($data);
    return htmlspecialchars(trim($data), ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');
}

function sanitize_array($array){
    foreach ($array as $k => $v){
        $array[$k] = is_array($v) ? sanitize_array($v) : sanitize($v);
    }
    return $array;
}

function clean($str){
    return htmlspecialchars(trim(strip_tags($str)), ENT_QUOTES, 'UTF-8');
}

function secure_input_all(){
    $_GET  = sanitize_array($_GET);
    $_POST = sanitize_array($_POST);
}

secure_input_all();



/* =======================================================
    3) حماية من Path Traversal
======================================================= */
function safe_path($path){
    // يمنع محاولات ../../
    $path = str_replace(['../','./','..\\','.\\'], '', $path);

    // منع المسار المُطلق
    if (strpos($path, '/') === 0 || strpos($path, '\\') === 0) {
        die("❌ محاولة وصول غير مسموحة للمسار");
    }

    return $path;
}



/* =======================================================
    4) CSRF Protection
======================================================= */

function csrf_generate(){
    if (!isset($_SESSION['csrf_token'])){
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    return $_SESSION['csrf_token'];
}

function csrf_verify_or_block(){
    if ($_SERVER["REQUEST_METHOD"] === "POST") {

        $token = $_POST['csrf_token'] ?? "";

        if (!isset($_SESSION['csrf_token']) || !hash_equals($_SESSION['csrf_token'], $token)){
            http_response_code(403);
            die("❌ CSRF Token غير صالح – العملية مرفوضة!");
        }
    }
}



/* =======================================================
    5) حماية SQL Injection
======================================================= */

function sql_safe($val){
    return addslashes(trim($val));
}



/* =======================================================
    6) Block Suspicious Requests
======================================================= */

function block_suspicious_requests()
{
    $patterns = [
        "<script", "onerror", "alert(", "drop table",
        "union select", "information_schema", "sleep(",
        "<?php", "../", "..\\", "base64_"
    ];

    $all = json_encode($_REQUEST);

    foreach ($patterns as $p) {
        if (stripos($all, $p) !== false) {
            die("❌ تم اكتشاف محاولة اختراق – تم حظر الطلب");
        }
    }
}

block_suspicious_requests();



/* =======================================================
    7) User Agent Check
======================================================= */

function validate_user_agent()
{
    if (!isset($_SERVER['HTTP_USER_AGENT']) || strlen($_SERVER['HTTP_USER_AGENT']) < 5) {
        die("❌ المتصفح غير معروف – تم منع الوصول");
    }
}

validate_user_agent();



/* =======================================================
    8) Referer Validation
======================================================= */

function validate_referer()
{
    if ($_SERVER["REQUEST_METHOD"] === "POST") {

        // إذا أتى الطلب من تيليجرام أو تطبيق خارجي قد لا يحتوي Referer → نتجاهله
        if (!isset($_SERVER["HTTP_REFERER"])) return;

        $domain = $_SERVER['HTTP_HOST'];

        if (strpos($_SERVER["HTTP_REFERER"], $domain) === false) {
            die("❌ طلب غير آمن (خطأ Referer)");
        }
    }
}

validate_referer();



/* =======================================================
    9) Anti-BruteForce Login Protection
======================================================= */

function login_attempt_block(){
    if (!isset($_SESSION['login_attempts'])) {
        $_SESSION['login_attempts'] = 0;
        $_SESSION['lock_time'] = null;
    }

    if ($_SESSION['login_attempts'] >= 5) {

        if (!$_SESSION['lock_time']) {
            $_SESSION['lock_time'] = time();
        }

        $remain = 60 - (time() - $_SESSION['lock_time']);
        if ($remain > 0) {
            die("🚫 تم إغلاق تسجيل الدخول لمدة: $remain ثانية");
        } else {
            $_SESSION['login_attempts'] = 0;
            $_SESSION['lock_time'] = null;
        }
    }
}

function login_attempt_fail(){
    $_SESSION['login_attempts'] = ($_SESSION['login_attempts'] ?? 0) + 1;
}

function login_attempt_success(){
    $_SESSION['login_attempts'] = 0;
    $_SESSION['lock_time'] = null;
}



/* =======================================================
    10) حماية رفع الملفات Upload Security
======================================================= */

function secure_file_upload($file, $allowed_ext = ['jpg','jpeg','png','pdf'])
{
    $name = $file['name'];
    $size = $file['size'];
    $tmp  = $file['tmp_name'];

    // تنظيف الاسم
    $safe_name = preg_replace("/[^A-Za-z0-9\._-]/", "", $name);

    // الامتداد
    $ext = strtolower(pathinfo($safe_name, PATHINFO_EXTENSION));

    if (!in_array($ext, $allowed_ext)) {
        die("❌ امتداد غير مسموح");
    }

    // حجم كبير
    if ($size > 20 * 1024 * 1024) { // 20MB
        die("❌ حجم الملف كبير جداً");
    }

    if (!is_uploaded_file($tmp)) {
        die("❌ ملف غير صالح");
    }

    return $safe_name;
}



/* =======================================================
    11) IP Lock (جلسة لا تعمل على IP آخر)
======================================================= */

function require_same_ip()
{
    if (!isset($_SESSION['client_ip'])) {
        $_SESSION['client_ip'] = $_SERVER['REMOTE_ADDR'];
    } else {
        if ($_SESSION['client_ip'] !== $_SERVER['REMOTE_ADDR']) {
            die("❌ تم تغيير الـ IP – الجلسة غير صالحة");
        }
    }
}

require_same_ip();



/* =======================================================
    12) Page Access Guards
======================================================= */

function guard_admin()
{
    if (!isset($_SESSION['admin_logged_in'])) {
        header("Location: ../admin/login.php");
        exit;
    }
}

function guard_finance()
{
    if (!isset($_SESSION['finance_user'])) {
        header("Location: ../finance/login.php");
        exit;
    }
}

?>