<?php
session_start();
if (!isset($_SESSION["finance_user"])) {
    header("Location: login.php");
    exit;
}

require_once "../config/db.php";
require_once "../config/app.php";
require_once "../config/whatsapp.php";

/* ============================================================
   (0) تحديد المحاسب الحالي
============================================================ */
$current_user_id = $_SESSION["finance_user"]["id"] ?? null;

/* ============================================================
   1) التأكد من الطلب
============================================================ */
if ($_SERVER["REQUEST_METHOD"] !== "POST") {
    die("❌ Invalid request.");
}

/* ============================================================
   2) استلام بيانات POST
============================================================ */
$student_id     = intval($_POST["student_id"] ?? 0);
$account_id     = intval($_POST["account_id"] ?? 0);
$amount         = intval($_POST["amount"] ?? 0);
$payment_type   = trim($_POST["payment_type"] ?? "");
$receipt_number = trim($_POST["receipt_number"] ?? "");
$notes          = trim($_POST["notes"] ?? "");

$discount       = intval($_POST["discount"] ?? 0);
$payment_plan   = trim($_POST["payment_plan"] ?? "دفعة واحدة");
$first_due      = $_POST["first_due_date"] ?? null;
$plan_note      = trim($_POST["plan_note"] ?? "");

/* ============================================================
   3) التحقق من القيم الأساسية
============================================================ */
if ($student_id <= 0 || $account_id <= 0) {
    die("❌ Student/Account error.");
}
if ($amount <= 0) {
    die("❌ Amount must be greater than zero.");
}

/* ============================================================
   4) جلب بيانات الحساب المالي
============================================================ */
$acc_stmt = $conn->prepare("SELECT * FROM accounts WHERE id = ? LIMIT 1");
$acc_stmt->execute([$account_id]);
$account = $acc_stmt->fetch(PDO::FETCH_ASSOC);

if (!$account) {
    die("❌ Account not found.");
}

$total_fee_original = intval($account["total_fee"]);
$old_paid           = intval($account["paid"]);
$old_remaining      = intval($account["remaining"]);
$old_plan           = $account["payment_plan"];

/* ============================================================
   5) حساب الخصم وتحديث total_fee
============================================================ */
$new_total_fee = $total_fee_original;

if ($discount > 0 && $discount <= 100) {
    $new_total_fee = intval($total_fee_original - ($total_fee_original * ($discount / 100)));
}

$update_acc = $conn->prepare("
    UPDATE accounts
    SET discount = ?, 
        total_fee = ?, 
        payment_plan = ?, 
        due_date = ?, 
        last_update = NOW()
    WHERE id = ?
");

$update_acc->execute([
    $discount,
    $new_total_fee,
    $payment_plan,
    $first_due,
    $account_id
]);

/* ============================================================
   6) رفع صورة الوصل (اختياري)
============================================================ */
$receipt_image = null;

if (!empty($_FILES["receipt_image"]["name"])) {

    $dir = "../uploads/receipts/";

    if (!is_dir($dir)) {
        mkdir($dir, 0777, true);
    }

    $ext = pathinfo($_FILES["receipt_image"]["name"], PATHINFO_EXTENSION);
    $file_name = "receipt_" . uniqid() . "." . $ext;

    if (move_uploaded_file($_FILES["receipt_image"]["tmp_name"], $dir . $file_name)) {
        $receipt_image = $file_name;
    }
}

/* ============================================================
   7) حفظ الدفعة في جدول payments + تسجيل المحاسب
============================================================ */
$insert_payment = $conn->prepare("
    INSERT INTO payments
    (student_id, account_id, amount, receipt_number, receipt_image, payment_type, notes, paid_at, finance_user_id)
    VALUES (?,?,?,?,?,?,?,NOW(),?)
");

$insert_payment->execute([
    $student_id,
    $account_id,
    $amount,
    $receipt_number,
    $receipt_image,
    $payment_type,
    $notes,
    $current_user_id
]);

/* ============================================================
   8) تحديث paid + remaining في accounts
============================================================ */
$new_paid      = $old_paid + $amount;
$new_remaining = $new_total_fee - $new_paid;
if ($new_remaining < 0) $new_remaining = 0;

$update_account_balance = $conn->prepare("
    UPDATE accounts
    SET paid = ?, remaining = ?, last_update = NOW()
    WHERE id = ?
");

$update_account_balance->execute([
    $new_paid,
    $new_remaining,
    $account_id
]);

/* ============================================================
   9) توليد جدول الأقساط إذا لم يكن موجود
============================================================ */
function getPlanCount($label) {
    return [
        "دفعة واحدة" => 1,
        "دفعتين"     => 2,
        "3 دفعات"    => 3,
        "4 دفعات"    => 4,
        "5 دفعات"    => 5,
        "6 دفعات"    => 6,
    ][$label] ?? 1;
}

$check_schedule = $conn->prepare("
    SELECT COUNT(*) FROM installment_schedule WHERE account_id = ?
");
$check_schedule->execute([$account_id]);
$has_schedule = $check_schedule->fetchColumn() > 0;

$installments_count = getPlanCount($payment_plan);
$installment_amount = intval($new_total_fee / $installments_count);

if (!$has_schedule && $first_due) {

    $insert_sched = $conn->prepare("
        INSERT INTO installment_schedule
        (account_id, installment_number, amount, due_date, status)
        VALUES (?,?,?,?,?)
    ");

    for ($i = 1; $i <= $installments_count; $i++) {
        $due_date = date("Y-m-d", strtotime("+".($i-1)." month", strtotime($first_due)));

        $insert_sched->execute([
            $account_id,
            $i,
            $installment_amount,
            $due_date,
            "pending"
        ]);
    }
}

/* ============================================================
  10) توزيع مبلغ الدفعة على جدول الأقساط
============================================================ */
$remaining_payment = $amount;

$get_sched = $conn->prepare("
    SELECT * FROM installment_schedule
    WHERE account_id = ?
    ORDER BY installment_number ASC
");
$get_sched->execute([$account_id]);
$schedules = $get_sched->fetchAll(PDO::FETCH_ASSOC);

foreach ($schedules as $sch) {

    if ($remaining_payment <= 0) break;

    $due  = intval($sch["amount"]);
    $paid = intval($sch["paid_amount"]);
    $gap  = $due - $paid;

    if ($gap <= 0) continue;

    $pay_now = min($remaining_payment, $gap);

    $new_paid_amount = $paid + $pay_now;
    $new_status = ($new_paid_amount >= $due) ? "paid" : "partial";

    $update_sched = $conn->prepare("
        UPDATE installment_schedule
        SET paid_amount = ?, paid_date = NOW(), status = ?
        WHERE id = ?
    ");

    $update_sched->execute([
        $new_paid_amount,
        $new_status,
        $sch["id"]
    ]);

    $remaining_payment -= $pay_now;
}

/* ============================================================
  11) تعيين الأقساط المتأخرة
============================================================ */
$mark_late = $conn->prepare("
    UPDATE installment_schedule
    SET status = 'late'
    WHERE account_id = ?
      AND status = 'pending'
      AND due_date < CURDATE()
");
$mark_late->execute([$account_id]);

/* ============================================================
  12) إرسال رسالة واتساب للطالب
============================================================ */
$get_student = $conn->prepare("
    SELECT full_name, phone 
    FROM students 
    WHERE id = ? LIMIT 1
");
$get_student->execute([$student_id]);
$student = $get_student->fetch(PDO::FETCH_ASSOC);

if ($student && !empty($student["phone"])) {

    $msg = "
🏛️ *جامعة النبراس الأهلية – قسم الحسابات*

تم تسجيل دفعة جديدة:

👤 *الطالب:* {$student['full_name']}
💰 *المبلغ:* " . number_format($amount) . " د.ع
🧾 *رقم الوصل:* {$receipt_number}
💳 *طريقة الدفع:* {$payment_type}
📅 *تاريخ الدفع:* " . date("Y-m-d H:i") . "

📘 *القسط الكلي بعد الخصم:* " . number_format($new_total_fee) . " د.ع
🟢 *المدفوع:* " . number_format($new_paid) . " د.ع
🔴 *المتبقي:* " . number_format($new_remaining) . " د.ع

مع التحية.
";

    sendWhatsAppMessage($student["phone"], $msg);
}

/* ============================================================
  13) الرجوع لصفحة الملف المالي
============================================================ */
header("Location: student_profile.php?id=".$student_id);
exit;

?>